/* $Id: arbsrc.c,v 1.5 1998/09/16 02:04:53 ericb Exp $ */
/* Copyright (C) 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* Test arb source problems.  This tests for two known problems with
   the X.03.26 substrate firmware and 0x302f source firmware.

   The first problem is apparently caused by doing e1432_set_amp_scale
   after preloading the first of the two buffers with the source in
   continuous arb mode.  For this problem, after the init measure the
   source takes just one more buffer worth of data, and then never
   again sets arbrdy.  It reports overreads forever after that.  This
   problem can be fixed by putting a short time delay AFTER the init
   measure.  This problem happens maybe one time in four if the
   e1432_set_amp_scale is done.  This problem happens only if using
   channel 2 of the source (channel 1 can be active or inactive, it
   makes no difference).

   The second problem is that sometimes, after pre-loading both A and
   B buffers, the source reports that it is not yet full and can
   accept more data.  This is before the init measure, so the source
   can really hold only two buffers worth of data.  If the error is
   ignored, and an init measure is done, the source takes only one
   more scan (?) and then stops, but does not report overreads (?).
   This problem is rare - maybe one in 20 or one in 50 measurements do
   this.  This problem happens when using either channel 1 or channel
   2 or both.

   This program needs to be told which source channels to use.  By
   default it uses none, so you must specify either "-1", "-2", or
   "-12".  A typical use would be "while ./arbsrc -2 -v; do :; done",
   which tests channel 2 of the source.  The while loop runs the
   program over and over until an error happens. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "e1432.h"

#define	SRCBUFFER_SIZE	1024
#define	NSCAN		10

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: arbsrc.c,v 1.5 1998/09/16 02:04:53 ericb Exp $";
static const char *progname;
static long overread_count, overread_last;

static int
check_src_status(E1432ID hw, int nchan, SHORTSIZ16 *chan_list)
{
    int     i, status;

    for (i = 0; i < nchan; i++)
    {
	/* For these functions, negative return value means error,
	   positive means the condition happenned.  Either way, we
	   want to know about it. */

	status = e1432_check_src_overload(hw, chan_list[i]);
	if (status != 0)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_check_src_overload: returned %d\n",
			   progname, status);
	    return -1;
	}

	/* Use CHECK to weed out actual errors */
	CHECK(status = e1432_check_src_overread(hw, chan_list[i]));
	if (status != 0)
	{
	    /* This can happen in continuous arb mode if the host
	       can't keep up with the data, or if the operating system
	       swaps us out for too long.  Print a warning
	       (sometimes), but don't abort until this has happenned
	       many times. */
	    overread_count++;
	    if (overread_count >= 2 * overread_last)
	    {
		overread_last = overread_count;
		(void) fprintf(stderr,
			       "%s: e1432_check_src_overread: %d returned %d\n",
			       progname, overread_count, status);
		if (overread_count >= 4096)
		    return -1;
	    }
	}

	status = e1432_check_src_shutdown(hw, chan_list[i]);
	if (status != 0)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_check_src_shutdown: returned %d\n",
			   progname, status);
	    return -1;
	}

	status = e1432_check_src_underrun(hw, chan_list[i]);
	if (status != 0)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_check_src_underrun: returned %d\n",
			   progname, status);
	    return -1;
	}
    }

    return 0;
}

static int
check_arbrdy(E1432ID hw, int nchan, SHORTSIZ16 *chan_list)
{
    int     i, status;

    for (i = 0; i < nchan; i++)
    {
	CHECK(status = e1432_check_src_arbrdy(hw, chan_list[i],
					      E1432_SRC_DATA_MODE_WAITAB));
	if (status == 0)
	    return 0;
    }
    return 1;
}

static int
wait_arbrdy(E1432ID hw, int chan)
{
    clock_t start, timeout;
    int     status;

    timeout = 2 * CLOCKS_PER_SEC;
    start = clock();
    while ((status = e1432_check_src_arbrdy(hw, chan,
					    E1432_SRC_DATA_MODE_WAITAB)) == 0)
	if (clock() - start > timeout &&
	    (status = e1432_check_src_arbrdy(hw, chan,
					     E1432_SRC_DATA_MODE_WAITAB)) == 0)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_check_src_arbrdy: "
			   "timeout waiting %g sec\n",
			   progname, (double) timeout / CLOCKS_PER_SEC);
	    return -1;
	}
    if (status < 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_check_src_arbrdy: returned %d\n",
		       progname, status);
	return -1;
    }

    return 0;
}

/* Fill one source buffer, either A or B, for all source channels we
   are using. */
static int
xfer_src_data(E1432ID hw, int nchan, SHORTSIZ16 *chan_list, LONGSIZ32 *buf)
{
    long    xfer, offset, npoint;
    int     i;

    /* We must loop through all channels doing only
       E1432_SRC_DATA_NUMWORDS_MAX at a time. */
    offset = 0;
    npoint = SRCBUFFER_SIZE;
    while (npoint > 0)
    {
	xfer = E1432_SRC_DATA_NUMWORDS_MAX;
	if (xfer > npoint)
	    xfer = npoint;

	for (i = 0; i < nchan; i++)
	{
	    CHECK(wait_arbrdy(hw, chan_list[i]));
	    CHECK(e1432_write_srcbuffer_data(hw, chan_list[i],
					     buf + offset, xfer,
					     E1432_SRC_DATA_MODE_WAITAB));
	}

	npoint -= xfer;
	offset += xfer;
    }

    return 0;
}

static int
init(SHORTSIZ16 laddr, E1432ID *hw, int *group,
     int ch1, int ch2, int *nchan, SHORTSIZ16 *chan_list)
{
    struct e1432_hwconfig hwconfig;
    int     limit;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(1, &laddr, hw));
    CHECK(e1432_get_hwconfig(1, &laddr, &hwconfig));

    limit = ch2 ? 2 : 1;
    if (hwconfig.source_chans < limit)
    {
	(void) fprintf(stderr, "%s: Need %d source channel(s)\n",
		       progname, limit);
	return -1;
    }

    *nchan = 0;
    if (ch1)
	chan_list[(*nchan)++] = E1432_SOURCE_CHAN(1);
    if (ch2)
	chan_list[(*nchan)++] = E1432_SOURCE_CHAN(2);

    *group = e1432_create_channel_group(*hw, *nchan, chan_list);
    if (*group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *group);
	return -1;
    }

    return 0;
}

static int
setup(E1432ID hw, int group)
{
    overread_count = 0;
    overread_last = 0;

    CHECK(e1432_set_active(hw, group, E1432_CHANNEL_ON));
    CHECK(e1432_set_clock_freq(hw, group, 51200));
    CHECK(e1432_set_range(hw, group, 10));
    CHECK(e1432_set_source_mode(hw, group, E1432_SOURCE_MODE_ARB));
    CHECK(e1432_set_span(hw, group, 20000));
    CHECK(e1432_set_srcbuffer_mode(hw, group, E1432_SRCBUFFER_CONTINUOUS));
    CHECK(e1432_set_srcbuffer_size(hw, group, SRCBUFFER_SIZE));
    CHECK(e1432_set_srcbuffer_init(hw, group, E1432_SRCBUFFER_INIT_EMPTY));

    return 0;
}

static int
run(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list, int verbose)
{
    LONGSIZ32 buf[SRCBUFFER_SIZE];
    int     i, status;

    /* Zero the data */
    (void) memset(buf, 0, sizeof buf);

    /* Fill A */
    if (verbose)
	(void) printf("Xfer scan 1 (A buffer)\n");
    CHECK(xfer_src_data(hw, nchan, chan_list, buf));

#if 1
    /* If we do this, then about one measurement in four will fail
       after the third scan.  A delay after e1432_init_measure will
       avoid the failure.  When the measurement fails, arbrdy is never
       set and the module just reports overreads over and over. */
    CHECK(e1432_set_amp_scale(hw, group, 0.1));
#endif

    /* Fill B */
    if (verbose)
	(void) printf("Xfer scan 2 (B buffer)\n");
    CHECK(xfer_src_data(hw, nchan, chan_list, buf));

    /* Check source status */
    CHECK(check_src_status(hw, nchan, chan_list));
    CHECK(status = check_arbrdy(hw, nchan, chan_list));
    if (nchan > 0 && status != 0)
    {
	(void) fprintf(stderr,
		       "%s: ready after sending B data before init measure?\n",
		       progname);
	return -1;
    }

    /* Start measurement */
    CHECK(e1432_init_measure(hw, group));

    /* Transfer the rest of the data */
    for (i = 0; i < NSCAN - 2; i++)
    {
	do
	{
	    CHECK(check_src_status(hw, nchan, chan_list));
	    CHECK(status = check_arbrdy(hw, nchan, chan_list));
	}
	while (status == 0);

	if (verbose)
	    (void) printf("Xfer scan %d\n", i + 3);
	CHECK(xfer_src_data(hw, nchan, chan_list, buf));
    }

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-12uvV] [-L laddr]\n"
		   "\t-1: Use source channel 1\n"
		   "\t-2: Use source channel 2\n"
		   "\t-L: Logical address is <laddr>, default 8\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Verbose output\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr;
    SHORTSIZ16 chan_list[2];
    char   *p;
    int     c, ch1, ch2, verbose, group, nchan;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    ch1 = 0;
    ch2 = 0;
    laddr = 8;
    verbose = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "12L:uvV")) != -1)
	switch (c)
	{
	case '1':
	    ch1 = 1;
	    break;
	case '2':
	    ch2 = 1;
	    break;
	case 'L':
	    laddr = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr < 0 || laddr > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    if (!ch1 && !ch2)
    {
	(void) fprintf(stderr, "%s: activate at least one source channel\n",
		       progname);
	usage();
    }

    /* Run the measurement */
    if (init(laddr, &hw, &group, ch1, ch2, &nchan, chan_list) < 0)
	return EXIT_FAILURE;
    if (setup(hw, group) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, chan_list, verbose) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
